package cn.com.grandage.ssedemo.controller;

import cn.com.grandage.ssedemo.dto.DifyRequest;
import cn.com.grandage.ssedemo.service.DifyService;
import jakarta.annotation.Resource;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;
import reactor.core.publisher.Flux;

import java.time.Duration;
import java.util.UUID;

@RestController
@RequestMapping("/api/dify")
@CrossOrigin(origins = "*")
public class DifyController {

    @Resource
    private DifyService difyService;

    /**
     * 纯文本流式输出接口
     */
    @PostMapping(value = "/chat/stream",
            produces = MediaType.TEXT_PLAIN_VALUE + ";charset=UTF-8",
            consumes = MediaType.APPLICATION_JSON_VALUE + ";charset=UTF-8")
    public Flux<String> streamChat(@RequestBody ChatRequest chatRequest) {
        System.out.println("接收到请求: " + chatRequest.getMessage());
        DifyRequest difyRequest = buildDifyRequest(chatRequest);

        System.out.println("构建的Dify请求: " + difyRequest);

        return difyService.streamChatContent(difyRequest)
                .doOnNext(content -> System.out.println("发送内容: " + content))
                .delayElements(Duration.ofMillis(20)); // 减少延迟以提高响应速度
    }


    /**
     * 健康检查接口
     */
    @GetMapping("/health")
    public String health() {
        return "服务正常运行中...";
    }

    /**
     * 构建Dify请求对象
     */
    private DifyRequest buildDifyRequest(ChatRequest chatRequest) {
        DifyRequest difyRequest = new DifyRequest();
        difyRequest.setQuery(chatRequest.getMessage());
        difyRequest.setUser(chatRequest.getUser() != null ?
                chatRequest.getUser() : "user-" + UUID.randomUUID());
        difyRequest.setConversationId(chatRequest.getConversationId());
        difyRequest.setResponseMode("streaming"); // 确保使用流式模式
        return difyRequest;
    }

    // 内部请求类
    public static class ChatRequest {
        private String message;
        private String user;
        private String conversationId;

        // 无参构造函数
        public ChatRequest() {
        }

        // Getters and Setters
        public String getMessage() {
            return message;
        }

        public void setMessage(String message) {
            this.message = message;
        }

        public String getUser() {
            return user;
        }

        public void setUser(String user) {
            this.user = user;
        }

        public String getConversationId() {
            return conversationId;
        }

        public void setConversationId(String conversationId) {
            this.conversationId = conversationId;
        }

        @Override
        public String toString() {
            return "ChatRequest{" +
                    "message='" + message + '\'' +
                    ", user='" + user + '\'' +
                    ", conversationId='" + conversationId + '\'' +
                    '}';
        }
    }
}