package cn.com.grandage.ssedemo.service;

import cn.com.grandage.ssedemo.dto.DifyRequest;
import cn.com.grandage.ssedemo.dto.DifyResponse;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Service;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.core.publisher.Flux;

import java.util.Objects;

@Service
public class DifyService {

    private WebClient webClient;
    private final ObjectMapper objectMapper;
    
    @Value("${dify.api.url}")
    private String DIFY_API_URL;
    
    @Value("${dify.api.key}")
    private String API_KEY;

    public DifyService() {
        this.objectMapper = new ObjectMapper();
    }

    /**
     * 初始化WebClient
     */
    private void initWebClient() {
        if (this.webClient == null) {
            this.webClient = WebClient.builder()
                    .baseUrl(DIFY_API_URL)
                    .defaultHeader(HttpHeaders.AUTHORIZATION, "Bearer " + API_KEY)
                    .defaultHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE)
                    .defaultHeader("User-Agent", "DifySSEClient/1.0")
                    .build();
        }
    }

    /**
     * 调用Dify流式API并返回消息内容流
     */
    public Flux<DifyStreamData> streamChat(DifyRequest request) {
        // 初始化WebClient
        initWebClient();
        
        return webClient.post()
                .uri("")
                .bodyValue(request)
                .retrieve()
                .bodyToFlux(String.class)
                .doOnNext(line -> System.out.println("原始响应行: " + line))
                .filter(line -> line != null && !line.trim().isEmpty())
                .mapNotNull(this::parseStreamData)
                .filter(Objects::nonNull)
                .doOnNext(data -> System.out.println("解析后数据: Event=" + data.getEvent() + ", Content=" + data.getContent()));
    }

    /**
     * 仅返回消息内容的流
     */
    public Flux<String> streamChatContent(DifyRequest request) {
        return streamChat(request)
                .doOnNext(data -> System.out.println("过滤前数据: Event=" + data.getEvent() + ", Content=" + data.getContent()))
                .filter(data -> "message".equals(data.getEvent()) || "message_end".equals(data.getEvent()))
                .filter(data -> data.getContent() != null || "message_end".equals(data.getEvent()))
                .map(data -> {
                    if ("message_end".equals(data.getEvent())) {
                        return "[DONE]"; // 结束标记
                    }
                    return data.getContent();
                })
                .filter(content -> content != null && !content.isEmpty())
                .doOnNext(content -> System.out.println("最终发送内容: " + content));
    }

    private DifyStreamData parseStreamData(String jsonData) {
        try {
            System.out.println("尝试解析JSON: " + jsonData);
            DifyResponse response = objectMapper.readValue(jsonData, DifyResponse.class);

            DifyStreamData streamData = new DifyStreamData();
            streamData.setEvent(response.getEvent());
            streamData.setTaskId(response.getTaskId());
            streamData.setMessageId(response.getMessageId());
            streamData.setConversationId(response.getConversationId());
            streamData.setCreatedAt(response.getCreatedAt());

            // 根据不同事件类型处理
            switch (response.getEvent()) {
                case "message":
                    streamData.setContent(response.getAnswer() != null ? response.getAnswer() : "");
                    System.out.println("Message事件内容: " + response.getAnswer());
                    break;
                case "message_end":
                    streamData.setMetadata(response.getMetadata());
                    streamData.setUsage(response.getUsage());
                    System.out.println("Message结束事件");
                    break;
                case "error":
                    streamData.setErrorStatus(response.getStatus());
                    streamData.setErrorCode(response.getCode());
                    streamData.setErrorMessage(response.getMessage());
                    System.out.println("错误事件: " + response.getMessage());
                    break;
                default:
                    // 其他事件类型也保存原始数据
                    System.out.println("其他事件: " + response.getEvent());
                    break;
            }

            return streamData;
        } catch (Exception e) {
            System.err.println("解析JSON数据时出错: " + jsonData + ", 错误: " + e.getMessage());
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 内部数据传输类
     */
    public static class DifyStreamData {
        private String event;
        private String taskId;
        private String messageId;
        private String conversationId;
        private String content;
        private Long createdAt;
        private Object metadata;
        private Object usage;
        private String errorStatus;
        private String errorCode;
        private String errorMessage;

        // Getters and Setters
        public String getEvent() {
            return event;
        }

        public void setEvent(String event) {
            this.event = event;
        }

        public String getTaskId() {
            return taskId;
        }

        public void setTaskId(String taskId) {
            this.taskId = taskId;
        }

        public String getMessageId() {
            return messageId;
        }

        public void setMessageId(String messageId) {
            this.messageId = messageId;
        }

        public String getConversationId() {
            return conversationId;
        }

        public void setConversationId(String conversationId) {
            this.conversationId = conversationId;
        }

        public String getContent() {
            return content;
        }

        public void setContent(String content) {
            this.content = content;
        }

        public Long getCreatedAt() {
            return createdAt;
        }

        public void setCreatedAt(Long createdAt) {
            this.createdAt = createdAt;
        }

        public Object getMetadata() {
            return metadata;
        }

        public void setMetadata(Object metadata) {
            this.metadata = metadata;
        }

        public Object getUsage() {
            return usage;
        }

        public void setUsage(Object usage) {
            this.usage = usage;
        }

        public String getErrorStatus() {
            return errorStatus;
        }

        public void setErrorStatus(String errorStatus) {
            this.errorStatus = errorStatus;
        }

        public String getErrorCode() {
            return errorCode;
        }

        public void setErrorCode(String errorCode) {
            this.errorCode = errorCode;
        }

        public String getErrorMessage() {
            return errorMessage;
        }

        public void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }

        @Override
        public String toString() {
            return "DifyStreamData{" +
                    "event='" + event + '\'' +
                    ", content='" + content + '\'' +
                    '}';
        }
    }
}